package ipsk.webapps.db.speech;

import java.net.URI;
import java.net.URISyntaxException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.Principal;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.UUID;

import javax.naming.NamingException;
import javax.persistence.EntityManager;
import javax.persistence.EntityTransaction;
import javax.persistence.Query;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Expression;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;

import org.jasypt.util.password.StrongPasswordEncryptor;
import org.jasypt.util.password.rfc2307.RFC2307SSHAPasswordEncryptor;

import edu.vt.middleware.password.AlphabeticalSequenceRule;
import edu.vt.middleware.password.LengthRule;
import edu.vt.middleware.password.NumericalSequenceRule;
import edu.vt.middleware.password.Password;
import edu.vt.middleware.password.PasswordData;
import edu.vt.middleware.password.PasswordValidator;
import edu.vt.middleware.password.QwertySequenceRule;
import edu.vt.middleware.password.RepeatCharacterRegexRule;
import edu.vt.middleware.password.Rule;
import edu.vt.middleware.password.RuleResult;
import ips.security.jaas.InetOrgPersonPrincipal;
import ipsk.beans.BeanModel;
import ipsk.beans.validation.ValidationResult;
import ipsk.db.speech.Account;
import ipsk.db.speech.Organisation;
import ipsk.db.speech.Person;
import ipsk.db.speech.Project;
import ipsk.db.speech.Speaker;
import ipsk.db.speech.UserRole;
import ipsk.db.speech.UserRoleId;
import ipsk.db.speech.UserRoleId.RoleName;
import ipsk.db.speech.account.AccountRequest;
import ipsk.db.speech.account.InvitationRequest;
import ipsk.net.SendMail;
import ipsk.net.SendMailException;
import ipsk.persistence.ParameterizedQuery;
import ipsk.persistence.QueryParam;
import ipsk.util.LocalizableMessage;
import ipsk.util.PasswordGenerator;
import ipsk.util.RadixConverters;
import ipsk.webapps.ControllerException;
import ipsk.webapps.ProcessResult;
import ipsk.webapps.ProcessResult.Type;


public class ProfileController extends BasicWikiSpeechController<Account> {
	public final static String DEFAULT_ADMIN_LOGIN="admin";
	public final static UserRoleId.RoleName DEFAULT_ADMIN_ROLE=UserRoleId.RoleName.ADMIN;
	public final static String DEFAULT_ADMIN_PASSWORD="corpus";
	
	public final static String PASSWORD_RESET_REQUEST_LIST_ATTR_NAME="passwordRequestManager";
	
	private Account account;
	private RFC2307SSHAPasswordEncryptor rfcPasswordEncryptor=null;
	private StrongPasswordEncryptor strongPasswordEncryptor=null;
	private MessageDigest sha5Digest=null;
	//private boolean showAccountName=false;
	private AccountRequest accountRequest=null;
	
	public ProfileController() {
		super("WebSpeechDBPU", Account.class, "account");
		// Profile controller has NO security manager
		//securityManager=new WikiSpeechSecurityManager(this);
		rfcPasswordEncryptor = new RFC2307SSHAPasswordEncryptor();
		strongPasswordEncryptor = new StrongPasswordEncryptor();
		try {
			// for Oracle JVM
			sha5Digest = MessageDigest.getInstance("SHA-512");
		} catch (NoSuchAlgorithmException e) {
			// for IBM JVM
			try {
				sha5Digest = MessageDigest.getInstance("SHA5");
			} catch (NoSuchAlgorithmException e1) {
				// ignore
			}
		}
	}

	public Account getById(String login) {
		return (Account)super.getById(login);
	}
	

	public boolean removable(Object bean) throws ControllerException {
		if(currentRequest !=null && bean instanceof Account) {
			Account oAcc=(Account)bean;
			Account reqAcc=getAccountByRequest(currentRequest);
			if(reqAcc!=null) {
				// protect own account for now
				String reqLogin=reqAcc.getLogin();
				String oAccLogin=oAcc.getLogin();
				if(!reqLogin.equals(oAccLogin)) {
					return super.removable(bean);
				}
			}
		}
		return false;
	}
	
	
	
	public void digestAndStorePassword(Account a,String plainTextPassword){
		EntityManager em = getThreadEntityManager();

		String rfcPasswd = rfcPasswordEncryptor.encryptPassword(plainTextPassword);
		a.setRfc2307Password(rfcPasswd);

		if(sha5Digest!=null){
			byte[] ba=plainTextPassword.getBytes();
			String digestedPassw=RadixConverters.bytesToHex(sha5Digest.digest(ba));
			a.setSha5HexPassword(digestedPassw);
		}

		String strongPasswd = strongPasswordEncryptor
				.encryptPassword(plainTextPassword);
		a.setStrongPassword(strongPasswd);
		em.merge(a);


	}
	
	
	private void digestPassword(Account a){
		EntityManager em = getThreadEntityManager();
		String plainTextPassword = a.getPassword();
		if (plainTextPassword != null && ! "".equals(plainTextPassword)) {
			String aRfcPw = a.getRfc2307Password();
			if (aRfcPw == null) {
				String rfcPasswd = rfcPasswordEncryptor.encryptPassword(plainTextPassword);
				a.setRfc2307Password(rfcPasswd);
				em.merge(a);
			}
			String sha5Password=a.getSha5HexPassword();
			if(sha5Digest!=null && sha5Password==null){
				byte[] ba=plainTextPassword.getBytes();
				String digestedPassw=RadixConverters.bytesToHex(sha5Digest.digest(ba));
				a.setSha5HexPassword(digestedPassw);
				em.merge(a);
			}
			String aStrongPw = a.getStrongPassword();
			if (aStrongPw == null) {
				String strongPasswd = strongPasswordEncryptor
						.encryptPassword(plainTextPassword);
				a.setStrongPassword(strongPasswd);
				em.merge(a);
			}
		}

	}
	
	public void digestPasswords() {
		EntityManager em = getThreadEntityManager();
		Query q = em.createQuery("SELECT object(o) FROM Account o");
//		q.setMaxResults(1);
		@SuppressWarnings("unchecked")
		List<ipsk.db.speech.Account> list = q.getResultList();
		for (Account a : list) {
			digestPassword(a);
		}
	}
	
	protected AccountRequest accountRequestByUUID(EntityManager em,UUID uuid) {

		AccountRequest ar=null;
		CriteriaBuilder cb = em.getCriteriaBuilder();
		CriteriaQuery<AccountRequest> cq = cb.createQuery(AccountRequest.class);
		Root<AccountRequest> qr = cq.from(AccountRequest.class);
		cq.select(qr);
		Predicate eqUuid=cb.equal(qr.get("uuid"),uuid);
		cq.where(eqUuid);
		TypedQuery<AccountRequest> aq = em.createQuery(cq);
		List<AccountRequest> arList=aq.getResultList();
		// should be unique
		if(arList.size()>0) {
			ar=arList.get(0);
		}
		return ar;
	}
	

	public void delete(String login) {

		EntityManager em = getEntityManager();
		EntityTransaction tx = null;
		try {
			tx = em.getTransaction();
			tx.begin();
			Account o = em.find(Account.class, login);
			em.remove(o);
			tx.commit();
		} catch (Exception e) {
			e.printStackTrace();
			if (tx != null && tx.isActive())
				tx.rollback();
		} finally {

			em.close();
		}
		// super.delete(Organisation.class,id);
	}
	

	public String suggestPassword() {
		PasswordGenerator passwdGen = new PasswordGenerator();
		return passwdGen.generateRandom();
	}
	
	public void login(HttpServletRequest req){
		currentRequest=req;
		String userPrincipalName=req.getUserPrincipal().getName();
		EntityManager em=getThreadEntityManager();
		account=em.find(Account.class,userPrincipalName);
		Account inetOrgAcc=getOrCreateInetOrgPerson();
		if(inetOrgAcc!=null){
			account=inetOrgAcc;
		}
	}
	
	public InetOrgPersonPrincipal getInetOrgPersonPrincipal(){
		if(currentRequest!=null){
			Principal up=currentRequest.getUserPrincipal();
			if(up instanceof ips.security.jaas.InetOrgPersonPrincipal){
				return (InetOrgPersonPrincipal)up;
			}
		}
		return null;
	}
	
	public static InetOrgPersonPrincipal getInetOrgPersonPrincipal(HttpServletRequest req){
		if(req!=null){
			Principal up=req.getUserPrincipal();
			if(up instanceof ips.security.jaas.InetOrgPersonPrincipal){
				return (InetOrgPersonPrincipal)up;
			}
		}
		return null;
	}
	
	public static Account getAccountInetOrgPerson(HttpServletRequest req,EntityManager em){
		
		// TODO duplicate code  getOrCreateInetOrgperson
		Account cAcc=null;
		InetOrgPersonPrincipal inetOrgPersonPrinc=getInetOrgPersonPrincipal(req);
		
		
		if(inetOrgPersonPrinc!=null){
			
			//  lookup corresponding JPA/SQL DB account
			String distinguishedName=inetOrgPersonPrinc.getName();
			
			// Fix for Bug ID WSP-0002 2019-02-22 closed
			// "Distinguished names of LDAP accounts were handled case sensitive but they should not"
			String distinguishedNameLc=distinguishedName.toLowerCase(Locale.ENGLISH);
			
			CriteriaBuilder cb = em.getCriteriaBuilder();
			CriteriaQuery<Account> cq = cb.createQuery(Account.class);
			Root<Account> qr = cq.from(Account.class);
			cq.select(qr);
			Expression<String> loginLc=cb.lower(qr.<String>get("login"));
			Predicate loginEqualsCaseInsensitive=cb.equal(loginLc, distinguishedNameLc);
			cq.where(loginEqualsCaseInsensitive);
			TypedQuery<Account> aq = em.createQuery(cq);
			List<Account> resultList = aq.getResultList();

			
			int accsSize=resultList.size();
			if(resultList.isEmpty() || accsSize==0){
				
			}else {
				cAcc=resultList.get(0);
//				if(accsSize>1) {
//					getServletContext().log("WARNING: "+accsSize+" account entries found for case insensitive query "+distinguishedName);
//					getServletContext().log("WARNING: Using first account entry: "+cAcc.getLogin());
//				}
				
			}
			
		}
		return cAcc;
	}

	
	
	public boolean copyOfInetOrgPersonRequired(){
		InetOrgPersonPrincipal inetOrgPersonPrinc=getInetOrgPersonPrincipal();
		if(inetOrgPersonPrinc!=null){
			//  lookup corresponding JPA/SQL DB account
			String distinguishedName=inetOrgPersonPrinc.getName();
			EntityManager em = getThreadEntityManager();
			
			Account cAcc=em.find(Account.class,distinguishedName);
			return(! (cAcc!=null));
		}
		return false;
	}
	
	public Account getOrCreateInetOrgPerson(){
		//Account newAcc=null;
		Account cAcc=null;
		InetOrgPersonPrincipal inetOrgPersonPrinc=getInetOrgPersonPrincipal();
		if(inetOrgPersonPrinc!=null){
			
			//  lookup corresponding JPA/SQL DB account
			String distinguishedName=inetOrgPersonPrinc.getName();
//			EntityManager em = getThreadEntityManager();
//			
//			Account cAcc=em.find(Account.class,distinguishedName);
//			if(cAcc==null){
//				cAcc=new Account(distinguishedName);
//				cAcc.setEditable(false);
//				em.persist(cAcc);
//				newAcc=cAcc;
//			}
//			
			// Fix for Bug ID WSP-0002 2019-02-22 closed
			// "Distinguished names of LDAP accounts were handled case sensitive but they should not"
			String distinguishedNameLc=distinguishedName.toLowerCase(Locale.ENGLISH);
			EntityManager em = getThreadEntityManager();

			CriteriaBuilder cb = em.getCriteriaBuilder();
			CriteriaQuery<Account> cq = cb.createQuery(Account.class);
			Root<Account> qr = cq.from(Account.class);
			cq.select(qr);
			Expression<String> loginLc=cb.lower(qr.<String>get("login"));
			Predicate loginEqualsCaseInsensitive=cb.equal(loginLc, distinguishedNameLc);
			cq.where(loginEqualsCaseInsensitive);
			TypedQuery<Account> aq = em.createQuery(cq);
			List<Account> resultList = aq.getResultList();

			
			int accsSize=resultList.size();
			if(resultList.isEmpty() || accsSize==0){

				cAcc=new Account(distinguishedNameLc);
				cAcc.setEditable(false);
				em.persist(cAcc);
				//newAcc=cAcc;
			}else {
				cAcc=resultList.get(0);
				if(accsSize>1) {
					getServletContext().log("WARNING: "+accsSize+" account entries found for case insensitive query "+distinguishedName);
					getServletContext().log("WARNING: Using first account entry: "+cAcc.getLogin());
				}

			}


//			Person ap=cAcc.getPerson();
//			if(ap==null){
//
//				Speaker p=new Speaker();
//
//				String foreName;
//				try {
//					foreName = inetOrgPersonPrinc.getAttrGivenname();
//					p.setForename(foreName);
//					String lastName=inetOrgPersonPrinc.getAttrSurname();
//					p.setName(lastName);
//				} catch (NamingException e) {
//					// TODO Auto-generated catch block
//					e.printStackTrace();
//				}
//				em.persist(p);
//
//				cAcc.setPerson(p);
//				em.merge(cAcc);
//
//				// Query all orgas with userDn ends with baseDn of Orga
//				// e.g. 'uid=test,ou=People,dc=example,dc=org' ends with 'dc=example,dc=org';
//				// -> LOCATE returns 20  = 54-34 -> match
//				Query q=em.createQuery("SELECT orga FROM Organisation orga WHERE orga.baseDN IS NOT NULL AND LOCATE(orga.baseDN,'"+distinguishedName+"') = (LENGTH('"+distinguishedName+"') - LENGTH(orga.baseDN) +1 )");
//
//				@SuppressWarnings("unchecked")
//				List<Organisation> maOrgas=q.getResultList();
//
//				//				CriteriaBuilder cb = em.getCriteriaBuilder();
//				//				Metamodel m = em.getMetamodel();
//				//
//				//				//						EntityType<Organisation> Organisation_ = m.entity(Organisation.class);
//				//				CriteriaQuery<Organisation> cq = cb.createQuery(Organisation.class);
//				//
//				//				Root<Organisation> orga = cq.from(Organisation.class);
//				//				//						EntityType<Organisation> Organisation_ = orga.getModel();
//				//				cq.select(orga);
//				//				cq.where(orga.get(Organisation_.baseDN).isNotNull());
//				//				cq.where(cb.like(orga.get(Organisation_.baseDN),"%"+distinguishedName));
//				//
//				//				List<Organisation> maOrgas = em.createQuery(cq).getResultList();
//				for(Organisation mOrga:maOrgas){
//					p.getOrganisations().add(mOrga);
//					mOrga.getPersons().add(p);
//					em.merge(p);
//					em.merge(mOrga);
//				}
//				//em.merge(cAcc);
//			}
			
		}
		return cAcc;
	}



	public Account getAccount() {
		return account;
	}

	public void setAccount(Account account) {
		this.account = account;
	}


	
	public static Account getAccountByRequest(HttpServletRequest req,EntityManager em){

		Account acc=null;
		if(req!=null){
			String accName=req.getRemoteUser();
			Principal userPrincipal=req.getUserPrincipal();
			if(userPrincipal!=null){
				String upNm=userPrincipal.getName();
				if(upNm!=null){
					accName=upNm;
				}
			}
			if(accName!=null){
				acc= em.find(Account.class,accName);
				if(acc==null) {
					acc=getAccountInetOrgPerson(req, em);
				}
				if(acc!=null) {
					em.refresh(acc);
				}
			}
		}
		return acc;
	}
	
	public static String getPasswordRequestManagerAttributeName(){
		return ProfileController.class.getName()+"."+PASSWORD_RESET_REQUEST_LIST_ATTR_NAME;
	}

	
	/**
	 * Sets encrypted account password.
	 * @param req the HTTP request, 
	 * @throws NoSuchAlgorithmException
	 */
	public void setPassword(HttpServletRequest req) throws ControllerException{
		ValidationResult vr=new ValidationResult();
		String passw1=req.getParameter("password1");
		String passw2=req.getParameter("password2");
		String login=req.getParameter("login");
		if(login==null){
			vr=new ValidationResult(ValidationResult.Type.ERRORS);		
		}else{
			EntityManager em=getThreadEntityManager();
			Account acc=em.find(Account.class,login);
			if(acc!=null && passw1 != null && passw2 !=null && passw1.equals(passw2)){	
				digestAndStorePassword(acc, passw1);
				this.account=acc;
				servletContext.log("Set password for login \""+acc.getLogin()+"\".");
			}else{
				vr=new ValidationResult(ValidationResult.Type.ERRORS);
				
			}
		}
		processResult=new ProcessResult(vr);
	}
	
	
	public boolean getShowAccountName(){
		if(accountRequest!=null){
			return(accountRequest instanceof InvitationRequest);
		}
		return false;
	}
	public void passwordRequest(HttpServletRequest req) throws NoSuchAlgorithmException, ControllerException{
		String cmd=processCommand(req,new String[]{"submit"});
		if("submit".equals(cmd)){
			passwordSetRequest(req);
		}else{
		accountRequest=null;
		setAccount(null);
		String uuidString=req.getParameter("uuid");
		if(uuidString!=null && ! uuidString.equals("")){
			UUID uuid=UUID.fromString(uuidString);
			//AccountRequestsManager pwrm=(AccountRequestsManager)servletContext.getAttribute(getPasswordRequestManagerAttributeName());
			EntityManager em=getThreadEntityManager();
			if(em!=null){
				
				AccountRequest pwrr=accountRequestByUUID(em, uuid);
				if(pwrr!=null){
					accountRequest=pwrr;
					String reqLogin=pwrr.getLogin();
					Account acc=getById(reqLogin);
					if(acc!=null){
						if(pwrr.isValid()){
							// OK

							account=acc;
						}else{
							servletContext.log("Expired password request for \""+account+"\" !");
							processResult=new ProcessResult(Type.ERROR);
							processResult.setResultKey("failed");
						}
					}else{
						servletContext.log("Password request: Corresponding account for login\""+reqLogin+"\" not found !");
						processResult=new ProcessResult(Type.ERROR);
						processResult.setResultKey("failed");
					}
				}else{
					servletContext.log("Password request: matching request not found!");
					processResult=new ProcessResult(Type.ERROR);
					processResult.setResultKey("failed");
				}
			}else{
				servletContext.log("Password request: matching request not found!");
				processResult=new ProcessResult(Type.ERROR);
				processResult.setResultKey("failed");
			}
		}
		if(account !=null){
			servletContext.log("Valid password request for \""+account+"\"");
		}
		}
	}
	
	public static boolean accountHasUserRole(Account acc,UserRoleId.RoleName userRoleName) {
		Set<UserRole> urs=acc.getUserRoles();
		for(UserRole ur:urs) {
			if(userRoleName.equals(ur.getId().getRoleName())) {
				return true;
			}
		}
		return false;
	}
	

	private PasswordValidator buildPasswordValidator() {
		// password check using vt-password library
		
		// password must be between 8 and 64 chars long
		LengthRule lengthRule = new LengthRule(8, 64);
		
		// don't allow alphabetical sequences
		AlphabeticalSequenceRule alphaSeqRule = new AlphabeticalSequenceRule();

		// don't allow numerical sequences of length 5
		NumericalSequenceRule numSeqRule = new NumericalSequenceRule(5, false);

		// don't allow qwerty sequences
		QwertySequenceRule qwertySeqRule = new QwertySequenceRule();

		// don't allow 4 repeat characters
		RepeatCharacterRegexRule repeatRule = new RepeatCharacterRegexRule(4);


		// group all rules together in a List
		List<Rule> ruleList = new ArrayList<Rule>();

		ruleList.add(lengthRule);
		ruleList.add(alphaSeqRule);
		ruleList.add(numSeqRule);
		ruleList.add(qwertySeqRule);
		ruleList.add(repeatRule);

		PasswordValidator validator = new PasswordValidator(ruleList);
		return validator;
	}
	
	public void passwordSetRequest(HttpServletRequest req) throws ControllerException{
		processResult=null;
		accountRequest=null;
		setAccount(null);
		String uuidString=req.getParameter("uuid");
		if(uuidString!=null && ! uuidString.equals("")){
			UUID uuid=UUID.fromString(uuidString);
			String passw1=req.getParameter("password1");
			String passw2=req.getParameter("password2");

			//AccountRequestsManager pwrm=(AccountRequestsManager)servletContext.getAttribute(getPasswordRequestManagerAttributeName());
			EntityManager em=getThreadEntityManager();
			if(em!=null){
				AccountRequest pwrr=accountRequestByUUID(em, uuid);
				if(pwrr!=null){
					accountRequest=pwrr;
					Account acc=getById(pwrr.getLogin());

					if(acc!=null){
						if(passw1 != null && passw2 !=null){
							if(passw1.equals(passw2)){
								if(!"".equals(passw1)){

									PasswordValidator validator=buildPasswordValidator();
									PasswordData passwordData = new PasswordData(new Password(passw1));
									RuleResult result = validator.validate(passwordData);

									if (result.isValid()) {
										// password check OK
										servletContext.log("Set password request for login \""+acc.getLogin()+"\"...");
										digestAndStorePassword(acc, passw1);
										this.account=acc;
										em.remove(pwrr);
										servletContext.log("Set password for login \""+acc.getLogin()+"\".");
										processResult=new ProcessResult(Type.SUCCESS);

									} else {
										// Bad password

										// TODO Password validator messages are not localized. Use RuleResult.getDetails() to create localized messages

										// default message
										String message="Password validation failed.";

										List<String> validatorMsgs=validator.getMessages(result);

										if(validatorMsgs.size()>0) {
											// simplay use the first one for now
											message=validatorMsgs.get(0);
										}

										processResult=new ProcessResult(Type.ERROR);
										LocalizableMessage lm=new LocalizableMessage(message);

										processResult.setLocalizableMessage(lm);
										processResult.setValidationResult(new ValidationResult(ValidationResult.Type.ERRORS));

									}
								}else{
									processResult=new ProcessResult(Type.ERROR);
									LocalizableMessage lm=new LocalizableMessage("Messages", "password.is_empty");

									processResult.setLocalizableMessage(lm);
									processResult.setValidationResult(new ValidationResult(ValidationResult.Type.ERRORS));

								}
							}else{
								processResult=new ProcessResult(Type.ERROR);
								LocalizableMessage lm=new LocalizableMessage("Messages", "password.passwords_not_equal");
								processResult.setLocalizableMessage(lm);
//								processResult.setResultKey("passwords_not_equal");
								processResult.setValidationResult(new ValidationResult(ValidationResult.Type.ERRORS));
							}
						}
					}else{
						processResult=new ProcessResult(Type.ERROR);
						processResult.setResultKey("failed");
						processResult.setLocalizableMessage(new LocalizableMessage("Could not find corresponding account!"));
					}
				}
			}
		}
	}
	
	public void loginSendRequest(HttpServletRequest req){
		String eMailParam=req.getParameter("email");
		if(eMailParam!=null && !eMailParam.equals("")){
			SendMail sm=new SendMail();
			if(sm.isAddressValid(eMailParam)){
				URI emailUri=null;
				try{
					emailUri=new URI(eMailParam);
				}catch(URISyntaxException use){
					servletContext.log("Login send request: E-Mail address \""+eMailParam+"\" malformed!", use);
					return;
				}
				if(emailUri!=null){
					List<Account> acclist=new ArrayList<Account>();
					EntityManager em=getThreadEntityManager();
				 
					//"SELECT ac FROM Account ac"
					CriteriaBuilder queryBuilder = em.getCriteriaBuilder();
					CriteriaQuery<Account> qdef = queryBuilder.createQuery(Account.class);
					Root<Account> accRoot = qdef.from(Account.class);
					qdef.select(accRoot);
					TypedQuery<Account> tq=em.createQuery(qdef);
					
					List<Account> allAccs=tq.getResultList();
					for(Account a:allAccs){
						String aEmail=a.contactEmail();
						if(aEmail!=null){
							try{
								URI aEmailUri=new URI(aEmail);
								if(emailUri.equals(aEmailUri)){
									acclist.add(a);

								}
							}catch(URISyntaxException use){
								// ignore
							}
						}
					}

					if(acclist.size()>0){
						// prepare E-Mail:
						// TODO localize!!
						StringBuffer contentBuffer=new StringBuffer("Dear WikiSpeech user!\n\nYour E-mail address is associated with the following accounts:\n\n");
						for(Account a:acclist){
							contentBuffer.append(a.getLogin()+"\n");
						}
						contentBuffer.append("\nPlease try to login:");
						// create the URL for the user to login
						String url=req.getScheme()+"://"+req.getServerName()+":"+req.getServerPort()+req.getContextPath()+"/login.jsp";
						contentBuffer.append(url.toString());
						contentBuffer.append("\n\nBest regards\n");
						String wikiSpeechEmail=servletContext.getInitParameter("wikispeech.email_address");
						if(wikiSpeechEmail==null){
							servletContext.log("Could not get WikiSpeech E-Mail (sender) address (\"wikispeech.email_address\" in web.xml)");
							return;
						}
						try {
							sm.send(wikiSpeechEmail, emailUri.toString(), "IPS Wikispeech Request", contentBuffer.toString());
							servletContext.log("Sent login request E-Mail from "+wikiSpeechEmail+" to "+emailUri.toString());
						} catch (SendMailException e) {
							servletContext.log("Could not send login request E-Mail",e);
							return;
						}
					}else{
						servletContext.log("Login send request: No account found for E-Mail address \""+eMailParam+"\"!");
					}
				}
			}
		}
	}
	
	@Override
	protected void add(HttpServletRequest request, String command)throws ControllerException {
		// store ...
		super.add(request,command);
		// ... and digest password
		BeanModel<Account> bm=getBeanModel();
		Account currAcc=bm.getBean();
		digestPassword(currAcc);
		
		
	}
	
	


}
