#!/usr/bin/perl

# #####################################################
#
# This file is part of the Perl scripts of the MASV System.
# MASV = Munich Automatic Speaker Verification
#
# Copyright 2002-2003, Ulrich Trk
# Institute of Phonetics and Speech Communication
# University of Munich
# tuerk@phonetik.uni-muenchen.de
#
#
#   MASV is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   MASV is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with MASV; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# #####################################################

my $CVS_Version_String = '$Id: preprocess_mlf.pl,v 1.6 2004/02/05 10:47:09 tuerk Exp $';
my $CVS_Name_String = '$Name: rel-1-4-01 $';


use strict;
my ($help, $Identify, $mlf_file, $mlf_handle, @mlf_file_content, $start, $end, %kan_lookup, $state_num, $current_model, $convert_statenames, $include_SD_pauses, $pool_name);
my ($current_speaker, $current_session, $current_recording);

use lib $ENV{"MASV_PERL_ROOT"};

use SR_lib;

use Getopt::Long qw( GetOptions );
use Pod::Usage;


$help = 0;
$Identify = 0;
$convert_statenames = '';
$include_SD_pauses = '';

GetOptions('convert_statenames=s' => \$convert_statenames,
           'include_SD_pauses=s' => \$include_SD_pauses,
           'help|?' => \$help,
           'version' => \$Identify);

if ($Identify) {
	printf "$0\n$CVS_Version_String\n$CVS_Name_String\n\n"; 
}

if ((@ARGV != 1) || ($help)) {
   pod2usage(1);
   exit 1;
 }
 

$mlf_file = $ARGV[0];

if ($include_SD_pauses) {
	$pool_name = $include_SD_pauses;
}

# read mlf
$mlf_handle = &SR_lib::open_file("<", "$mlf_file");
@mlf_file_content = <$mlf_handle>;
close $mlf_handle;

if ($convert_statenames) {
	&read_kan($convert_statenames);
}

#print STDOUT $kan_lookup{dreissig}[3] , "\n";
#exit 0;



# test third line, it must be generated with HVite and the option -f
# sample line (the last entries are not always present
# 0 1100000 s2 758.001526 silStart 4851.992676
my @test_array=split( /\s/, $mlf_file_content[2]);
if (! ($test_array[0] =~ /^\d+$/) &&
	($test_array[1] =~ /^\d+$/) &&
	($test_array[2] =~ /^s\d+$/) &&
	($test_array[3] =~ /^\d+\.\d+$/)  ) {
		die ("not the correct mlf input format\n");
}

	


my ($needs_begin_padding, $needs_end_padding, $rec_length, $pad_begin_length);
my ($start_time_utt, $end_time_utt);
my $new_end_time = 0;

$needs_begin_padding = 0;
$needs_end_padding = 0;
$rec_length = 0;
$pad_begin_length = 0;
$start_time_utt = 0;
$end_time_utt = 0;

foreach my $line (@mlf_file_content) {
	
	if ($line =~ /^\d+\s+\d+/) {
		# line with content
		my @line_contents = split( /\s/, $line);
		$state_num = $line_contents[2];
		$state_num =~ s/s//;
		$state_num--;
		$state_num--;
		
		if (defined($line_contents[4] )) {
			$current_model = $line_contents[4];
		}
		
		if ($convert_statenames) {
			my $new_statename = $kan_lookup{$current_model}[$state_num];
			$line =~ s/(\s+)s\d+(\s+)/${1}${new_statename}${2}/;
		}	
		if ($include_SD_pauses) {
			my $current_start_time = $line_contents[0];
			my $current_end_time = $line_contents[1];
			my $new_start_time = $current_start_time + $pad_begin_length;
			$new_end_time = $current_end_time + $pad_begin_length;
			if ($needs_begin_padding) {
				print STDOUT "0 $pad_begin_length SD_begin 0.0 SD_begin 0.0\n";
				$needs_begin_padding = 0;
			}
			$line =~ s/^${current_start_time}(\s+)/${new_start_time}${1}/;
			$line =~ s/(\s+)${current_end_time}(\s+)/${1}${new_end_time}${2}/;
		}
		print STDOUT $line;
	}
	elsif ($line =~ /^".*\/(\d{4})\/(\d{2})\/(.*)"/ ) {
		# filename of recording
		$current_speaker = $1;
		$current_session = $2;
		$current_recording = $3;
		$current_recording =~ s/\..*//;
		$needs_begin_padding = 0;
		$needs_end_padding = 0;
		
		if ($include_SD_pauses) {
			my $raw_file = $SR_lib::source_wav_files . $current_speaker . $SR_lib::sign . $current_session . $SR_lib::sign . $current_recording . ".${MASV_db_desc::audio_file_ext}";
			$rec_length = int((stat($raw_file))[7] / $MASV_db_desc::sampling_freq * 10000000); # length in 100 ns units
			my $info_file = $SR_lib::paramPool_dir . $pool_name . $SR_lib::sign . "SD_info"  . $SR_lib::sign . $current_speaker . $SR_lib::sign . $current_session . $SR_lib::sign . $current_recording . ".SD_info";
			($start_time_utt, $end_time_utt) = &get_SD_borders($info_file);
			#print "rec. " . $raw_file . "\n";
			#print "  " . $start_time_utt . " " . $end_time_utt . " " . $rec_length . "\n";
			$pad_begin_length = $start_time_utt;
			if ($start_time_utt > 0) {
				$needs_begin_padding = 1;
			}
			if ($end_time_utt < $rec_length) {
				$needs_end_padding = 1;
			}
			
		}
		print STDOUT $line;

	}
	elsif ($line =~ /^\.$/ ) {
		if ($include_SD_pauses) {
			if ($needs_end_padding) {
				print STDOUT "$new_end_time $rec_length SD_end 0.0 SD_end 0.0\n";
				$needs_end_padding = 0;
			}
		}
		print STDOUT $line;
	}
	else {
		# normal line 
		print STDOUT $line;
	}
	

}



exit 0;

sub get_SD_borders {
	
	my $SD_info_file = $_[0];
	
	my $info_handle = &SR_lib::open_file("<", "$SD_info_file");
	my @info_content = <$info_handle>;
	close $info_handle;
	
	my @elements = split(' ', $info_content[0]);
	my $time_start=int( ($elements[1] - 1) / $MASV_db_desc::sampling_freq * 10000000);
	
	@elements = split(' ', $info_content[1]);
	my $time_end=int( ($elements[1] + 1) / $MASV_db_desc::sampling_freq * 10000000);
	
	
	
	return ($time_start, $time_end);
	
}



sub read_kan {

	my $kan_filename = $_[0];
	my ($kan_handle, @kan_content);
	
	# read kan list
	$kan_handle = &SR_lib::open_file("<", "${SR_lib::base_dir}/hmm_lists/$kan_filename");
	@kan_content = <$kan_handle>;
	close $kan_handle;
	
	foreach my $line (@kan_content) {
		$line =~ /^([^\t]+)\t+(\d)\t+([^\t]+)/;
		my $word=$1;
		my $num_phonemes=$2;
		my $phonemes=$3;
		#print STDOUT ">>>>$word . $num_phonemes . $phonemes\n";
		my @phoneme_array = split('\" \"', $phonemes);
		if ((scalar @phoneme_array) != $num_phonemes) {
			die("error in $kan_filename: $phonemes should contain $num_phonemes phonemes !\n\n");
		}
		foreach my $element_id (0 .. $#phoneme_array) {
			$phoneme_array[$element_id] =~ s/ /\_/g;
			$phoneme_array[$element_id] =~ s/\"//g;
		}
		$kan_lookup{$word} = [ @phoneme_array ];
	}

}

	

__END__

=head1 NAME

preprocess_mlf.pl  -  do various processing with mlf files. Sends output to stdout.

=head1 SYNOPSIS

preprocess_mlf.pl  [options]  mlf_file

 Options:

 -convert_statenames=kanfile    change state names of "HVite -f" to phoneme names 
                                defined in ${MASV_BASEDIR}/hmm_lists/<kanfile> 

 -include_SD_pauses=pool_name   read SD information of pool <poolname> and 
                                add silence parts before and after the state string
                                if necessary.
 

 -version                       print version information.

 -? | help                      display this message.
 

=cut

