#!/usr/bin/perl

# #####################################################
#
# This file is part of the Perl scripts of the MASV System.
# MASV = Munich Automatic Speaker Verification
#
# Copyright 2002-2003, Ulrich Trk
# Institute of Phonetics and Speech Communication
# University of Munich
# tuerk@phonetik.uni-muenchen.de
#
#
#   MASV is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   MASV is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with MASV; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# #####################################################

my $CVS_Version_String = '$Id: rawal2waval.pl,v 1.9 2004/04/28 15:18:19 tuerk Exp $';
my $CVS_Name_String = '$Name: rel-1-4-01 $';

use strict;

use FileHandle;
use Getopt::Long qw( GetOptions );
use Pod::Usage;

my $help = 0;
my $Identify = 0;

my $sampling_frequency = 8000;
my $min_part_length = 200; # frame length normally here: 25ms =  200 samples
my $cut_const_parts = 0;
my $cut_down_length = 100; # length of const level part in output file


my $dither_const_parts = 0;

GetOptions('fs|sampling_freq=i' => \$sampling_frequency,
           'min_part_length=i' => \$min_part_length,
		   'cut_const_parts' => \$cut_const_parts,
           'dither_const_parts=i' => \$dither_const_parts,		   
		   'help|?' => \$help,
           'version' => \$Identify);

if ($Identify) {
	printf "$0\n$CVS_Version_String\n$CVS_Name_String\n\n"; 
}

if ((@ARGV != 2) || ($help)) {
   print "Usage: raw2wav.pl [options] <infile> <outfile>\n";
   print "   options:\n";
   print "     -fs : set sampling frequency; default 8000\n";
   print "     -min_part_length : set length of a constant level \n";
   print "      part to be processed (default: 200)\n";
   print "     -cut_const_parts : cut out const level part\n";
   print "     -dither_const_parts=i : add dither noise with scale i to const level parts\n\n";
   exit 1;
 }

if ($cut_const_parts && $dither_const_parts) {
	print "Options -cut_const_parts and -dither_const_parts=i are mutually exclusive!\n\n";
	exit 1;
}


my $raw_file = $ARGV[0];
my $wav_file = $ARGV[1];

###########################
my $display = 0;


# counters
if (!($cut_const_parts || $dither_const_parts)) {

	my $total_raw_length = -s "$raw_file";
	my $total_wav_length = $total_raw_length + 44;
	my $total_wav_chunk = $total_wav_length - 8;
	
	my $fh = &open_file("", "+>$wav_file");
	
	&write_wav_header($fh, $total_wav_chunk, $total_raw_length);
	
	&dump_raw_file_content("$raw_file", $fh);
	
	close ($fh);
	
}
elsif ($cut_const_parts) {
	
	my $offset_ref;
	my $length_ref;
	my $raw_data_ref;
	my $new_raw_data_ref;
	
	
	($offset_ref, $length_ref, $raw_data_ref) = &find_const_parts( $raw_file, $min_part_length );
	
	if ( (scalar (@{$offset_ref})) > 0 ) { # something there to cut out
		$new_raw_data_ref = &cut_data( $offset_ref, $length_ref, $raw_data_ref);
	}
	else {
		$new_raw_data_ref = $raw_data_ref;
	}
	
	my $total_raw_length = scalar @{$new_raw_data_ref};
	my $total_wav_length = $total_raw_length + 44;
	my $total_wav_chunk = $total_wav_length - 8;
	
	
	

	my $fh = &open_file("", "+>$wav_file");
	if ($display) {
	printf("chunk length: %08x\n",$total_wav_chunk);
	}
	
	&write_wav_header($fh, $total_wav_chunk, $total_raw_length);
	
	&dump_raw_data($fh, $new_raw_data_ref);
	
	close ($fh);
	
}
elsif ($dither_const_parts) {
	print "not implemented yet!\n\n";
}

exit 0;


sub write_wav_header {
	
	my $fh;
	my $total_wav_chunk;
	my $total_raw_length;
	
	($fh, $total_wav_chunk, $total_raw_length) = @_;
	
	&write_4c_to_file($fh, "RIFF", 4);
	&write_4b_to_file($fh, $total_wav_chunk, 4);
	&write_4c_to_file($fh, "WAVE", 4);
	&write_4c_to_file($fh, "fmt ", 4);
	&write_4b_to_file($fh, 16, 4);
	&write_2b_to_file($fh, 6, 2);
	&write_2b_to_file($fh, 1, 2);
	&write_4b_to_file($fh, $sampling_frequency, 4);
	&write_4b_to_file($fh, $sampling_frequency, 4);
	&write_2b_to_file($fh, 1, 2);
	&write_2b_to_file($fh, 8, 2);
	&write_4c_to_file($fh, "data", 4);
	&write_4b_to_file($fh, $total_raw_length, 4);
	
}

sub dump_raw_data {

	my $fh;
	my $data_ref;
	
	($fh, $data_ref) = @_;
	
	my $data = pack("C*", @{$data_ref});
	my $byte_num = scalar @{$data_ref};
	
	syswrite ($fh, $data, $byte_num);
	
	
}




sub cut_data {
	
	my $offset_ref = $_[0];
	my $length_ref = $_[1];
	my $data_ref = $_[2];
	
	my $begin_copy;
	my $end_copy;
	
	my $new_data_ref;
	@{$new_data_ref} = ();
	
	my @temp_array;
	
	$begin_copy = 0;
	$end_copy = (${$offset_ref}[0] - 1);
	
	if ($begin_copy <= $end_copy) { # add data before first const level part
		@temp_array = @{$data_ref}[ $begin_copy .. $end_copy ];
		push( @{$new_data_ref} , @temp_array);
	}
	
	if ( $#{$offset_ref} > 0 ) { # more than one part
		foreach my $part_counter (0 .. ($#{$offset_ref} -1)) { # all intermediate parts
			
			# cut const part
			$begin_copy = ${$offset_ref}[ $part_counter ];
			$end_copy = $begin_copy + $cut_down_length; # add only $cut_down_length no. of samples of const level part
			@temp_array = @{$data_ref}[ $begin_copy .. $end_copy ];
			push( @{$new_data_ref} , @temp_array);

			# part inbetween const parts
			$begin_copy = ${$offset_ref}[ $part_counter ] + ${$length_ref}[ $part_counter ];
			$end_copy = ${$offset_ref}[ $part_counter + 1 ] - 1;
			@temp_array = @{$data_ref}[ $begin_copy .. $end_copy ];
			push( @{$new_data_ref} , @temp_array);
			
		}
	
	}
	
	$begin_copy = ${$offset_ref}[ -1 ] + ${$length_ref}[ -1 ];
	$end_copy = $#{$data_ref};
	if ($begin_copy <= $end_copy) {
		@temp_array = @{$data_ref}[ $begin_copy .. $end_copy ];
		push( @{$new_data_ref} , @temp_array);
	}
	
	
	return $new_data_ref;
	
}


sub find_const_parts {
	
	my $current_rec = $_[0];
	my $min_part_length = $_[1];
	
	my $data_str;
	my @offset_array;
	my @length_array;
	
	my $offset_ref;
	my $length_ref;
	
	my $raw_file_length = -s $current_rec;

	
	my $raw_fh = &open_file("<", $current_rec);
	binmode ($raw_fh);
	my $length_read = read ($raw_fh, $data_str, $raw_file_length);
	close ($raw_fh);
	
	if ($length_read != $raw_file_length) {
		printf ("error when reading file $current_rec !\n");
		exit 1;
	}
	my @data = unpack("C*", $data_str);
	
	my $current_level_counter = 0;
	my $prev_level = $data[0] - 1;
	my $offset = 0;
	foreach my $data_value (@data) {
		if ($prev_level == $data_value) {
			$current_level_counter++;
		}
		else {
			if ($current_level_counter >= $min_part_length) {
				 push(@offset_array, ($offset-$current_level_counter)); # $offset here: first sample with different value, offset_array contains start of const level part 
				 push(@length_array, $current_level_counter);
			}
			$current_level_counter = 0;
		}
		$prev_level = $data_value;
		$offset++;
	}
	$offset_ref = [ @offset_array ];
	$length_ref = [ @length_array ];
	
	return ($offset_ref, $length_ref, [ @data ]);
}


sub open_file {
	my ($filemode, $filename) = @_;
	my $filevar = new FileHandle;
	open ($filevar, $filemode . $filename) ||  die ("Can't open $filename");
	binmode ($filevar);
	return $filevar;
}

sub read_from_file {
	my ($filevar, $byte_num) = @_;
	my $result;
	my $length = read ($filevar, $result, $byte_num);
	return ($result,$length);
}

sub write_4b_to_file {
	my ($filevar, $data, $byte_num) = @_;
	# check if 4 bytes 
	if ($byte_num == 4) {
		#$data = (($data & 0x0000ffff) << 16 ) | (($data & 0xffff0000) >> 16);
		my $data = pack("L", $data);
		syswrite ($filevar, $data, $byte_num);
	}
}

sub write_2b_to_file {
	my ($filevar, $data, $byte_num) = @_;
	# check if 2 bytes 
	if ($byte_num == 2) {
		my $data = pack("S", $data);
		syswrite ($filevar, $data, $byte_num);
	}
}

sub write_4c_to_file {
	my ($filevar, $data, $byte_num) = @_;
	# check if 4 bytes 
	if ($byte_num == 4) {
		#$data = substr($data,1,1) . substr($data,0,1) . substr($data,3,1) . substr($data,2,1);
		my $data = pack("A4", $data);
		syswrite ($filevar, $data, $byte_num);
	}
}


#
sub dump_raw_file_content {
	my ($filename_in, $file_out_var) = @_;
	my $block_size = 1024;
	my $fh = &open_file("", $filename_in);
	my ($data, $length);
	do 
	{
		($data, $length) = &read_from_file($fh,$block_size);
		syswrite ($file_out_var, $data, $length);
	} while ($length == $block_size);
	close ($fh);
}

