#!/usr/bin/perl


# #####################################################
#
# This file is part of the Perl scripts of the MASV System.
# MASV = Munich Automatic Speaker Verification
#
# Copyright 2002-2003, Ulrich Trk
# Institute of Phonetics and Speech Communication
# University of Munich
# tuerk@phonetik.uni-muenchen.de
#
#
#   MASV is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   MASV is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with MASV; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# #####################################################

my $CVS_Version_String = '$Id: run_HInit.pl,v 1.20 2004/03/09 21:13:46 tuerk Exp $';
my $CVS_Name_String = '$Name: rel-1-4-01 $';


use lib $ENV{"MASV_PERL_ROOT"};

use SR_lib;
use strict;

use File::Copy;
use File::Path;
use Getopt::Long qw( GetOptions );
use Pod::Usage;

use Parallel::ForkManager;

my ($trainingModelArrRef, $trainingModelSessionsList);

my $exit_state = 0;
my $local_exit_code = 0;
my @ssh_stat = [];

my $nice_factor = $SR_lib::nice_factor;
$nice_factor = $SR_lib::nice_factor; # make compiler happy, no warning

my $start_time = scalar localtime;

###########################
# Default values for options
my $help = 0;
my $log_file = '';
my $Identify = 0;
my $minVar = 0.0001;
my $buildWorldModel=0;
my $mlf = "mlf_prompted.mlf";
my $trainingModelConf = "training_set/at";
my $trainingWorldConf = "world_set/aa";
my $extraOptions = '';
my $protoDir = '';
my $hmmBaseDir = '';
my $distributedClient = 0;
my $totalHostsNum=0;
my $onStartup = '';
my $onFinish = '';
my $prefixCommand = '';
my $parallelComputation = 0;
my $useGMM = 0;
###########################
my $extraOptionsString;
my $useGMMString;

my ($hostsNum, @hostList);
my ($realHostsNum, @dummy);
my @exe_models;

my %prototypesType;
my @protos_exe;

my $prompt_HTK_Command_String = ($SR_lib::verbose_level > 1) ? "-A" : "";

# Processing Options

# flat
GetOptions('v|varFloor=f' => \$minVar, 
           'w|world' => \$buildWorldModel,
	       'm|mlf=s' => \$mlf,
		   'useGMM' => \$useGMM,
	       'tc|trainingModelConf=s' => \$trainingModelConf,
	       'wl|world_list=s' => \$trainingWorldConf,
	       'protoDir=s' => \$protoDir,
	       'hmmBaseDir=s' => \$hmmBaseDir,
		   'xo|extraOptions=s' => \$extraOptions,
           'dist=s' => \$distributedClient,
           'totalHostsNum=i' => \$totalHostsNum,
		   'onStartup=s' => \$onStartup,
		   'onFinish=s' => \$onFinish,
		   'prefixCommand=s' => \$prefixCommand,
           'p|parallelComputation' => \$parallelComputation,
           'help|?' => \$help,
		   'logfile=s' => \$log_file,
           'version' => \$Identify);

if ($Identify) {
	print STDOUT "$0\n$CVS_Version_String\n$CVS_Name_String\n\n"; 
}

if ((@ARGV != 1) || ($help)) {
   pod2usage(1);
   exit 1;
}

if ($SR_lib::verbose_level) {print STDOUT "Running $0 ...\n";}

if ($distributedClient && $parallelComputation) {
	print STDERR "Use only the -p|parallelComputation option!\n\n";
	exit 1;
}
 

my $sv_system_name = $ARGV[0];

&SR_lib::init_speaker_sets($sv_system_name);


$prefixCommand =~ s/[;\s]*$/;/ if ($prefixCommand);

my $prefixCommandString = ($prefixCommand) ? "-prefixCommand=\'$prefixCommand\'" : "";


$extraOptionsString = ($extraOptions) ? "-xo=\'$extraOptions\'"  :  "";
$useGMMString = ($useGMM) ? "-useGMM"  :  "";
if ($useGMM) {
	$mlf = "mlf_gmm.mlf";
}

if (!($hmmBaseDir)) {
	$hmmBaseDir = "hmm";
}
if (!($protoDir)) {
	print STDERR "No prototype path set! Use option protoDir!\n\n";
	exit 1;
}


($trainingModelArrRef, $trainingModelSessionsList) =  &SR_lib::processConfigurations($trainingModelConf);

my $sv_system_path = "${SR_lib::sv_systems_dir}${sv_system_name}$SR_lib::sign";
my $proto_path = "${sv_system_path}$protoDir";
my $world_path = "${sv_system_path}world$SR_lib::sign";
my $log_path = "${sv_system_path}log$SR_lib::sign";
$log_file = ($log_file) ? $log_file : "log.txt";




my @models = @{$trainingModelArrRef};


(opendir SRCDIR, $proto_path) || (die "cannot find dir: $proto_path");
my @files = grep(!/^\./, readdir(SRCDIR)); #Forget about . files
closedir SRCDIR;

# alle Macros aus der Liste entfernen
my @protos = ();
foreach my $file (@files) {
	if ($file !~ /macro/) {
		push @protos, $file;
	}
}

# Hash prototypesType mit num of states als key erzeugen
# Hash Eintrge zeigen auf Array mit Liste der dazugehrigen prototypes
foreach my $prototype (@protos) {
	my $states_in_prototype = &SR_lib::read_hmm_prototype($proto_path. $SR_lib::sign. $prototype);
	push (@{$prototypesType{$states_in_prototype}}, $prototype);	
}

# @protos_exe enhlt von jedem Typus den ersten Prototypen
foreach my $statesType (keys %prototypesType) {
	push (@protos_exe, $prototypesType{$statesType}->[0]);
}


if (! $buildWorldModel) {

	if ($parallelComputation) {
	
		($hostsNum, @hostList) = &SR_lib::par_get_available_hosts;
		
		# Wieviele hosts machen Sinn?
		($realHostsNum, @dummy) = &SR_lib::split_arrayList(\@models, $hostsNum);
			
	}
	elsif ($distributedClient) {
		# speaker-Liste auf bentigte Zahl aufteilen	
		my ($dummy, @models_lists) = &SR_lib::split_arrayList(\@models, $totalHostsNum);
		@exe_models = @{$models_lists[$distributedClient - 1]};	
	}
	else
	{#lokal ausgefhrt, alles lassen, wie es ist
		@exe_models = @models;
	}

}


if (! $distributedClient) {
	if ($onStartup) {
		if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onStartup\" ...\n");}
		$local_exit_code = system($onStartup);
	}
}


if (! $buildWorldModel) {
# seperate Modelle

	if ($parallelComputation) {
		my $pm = new Parallel::ForkManager($realHostsNum);

		$pm->run_on_finish(
			sub { my ($pid, $exit_code, $ident) = @_;
				  $ssh_stat[$ident] = $exit_code;
			}
		);
		
		for (my $host_counter=1; $host_counter <= $realHostsNum; $host_counter++) {
			$pm->start($host_counter) and next;
			
			if ($SR_lib::verbose_level) { print STDOUT ("Starting process on host $hostList[$host_counter]\n");}
			if ($onStartup) {
				if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onStartup\" ...\n");}
				$local_exit_code = system("ssh -x $hostList[$host_counter] \"$onStartup\"");
			}
			my $exec_string = "ssh -x $hostList[$host_counter] /usr/bin/nice -n $nice_factor \"run_HInit.pl -dist=$host_counter $useGMMString -v $minVar -tc=$trainingModelConf -mlf=$mlf -hmmBaseDir=$hmmBaseDir $extraOptionsString $prefixCommandString -totalHostsNum=$realHostsNum $sv_system_name \"";
			if ($SR_lib::verbose_level) { print STDOUT "$exec_string \n";}
			$local_exit_code += system($exec_string);
			if ($onFinish) {
				if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onFinish\" ...\n");}
				$local_exit_code += system("ssh -x $hostList[$host_counter] \"$onFinish\"");
			}
			if ($local_exit_code) {
				$local_exit_code = 11;
			}
			
			if ($SR_lib::verbose_level) { print STDOUT "Host $hostList[$host_counter], status: $local_exit_code \n";}
			if ($SR_lib::verbose_level) { print STDOUT ("Process on  $hostList[$host_counter] complete.\n");}

			$pm->finish($local_exit_code);
	
		}
	  
		$pm->wait_all_children;
		if ($SR_lib::verbose_level) { print STDOUT ("All processes completed.\n");}
		if ($SR_lib::verbose_level) { print STDOUT ("Process status summary:\n");}
		
		for (my $host_counter=1; $host_counter <= $hostsNum; $host_counter++) {
			if ($SR_lib::verbose_level) { print STDOUT "Host $hostList[$host_counter], status: $ssh_stat[$host_counter] \n";}
			$exit_state += $ssh_stat[$host_counter];
		}
	
		if ($SR_lib::verbose_level) { print STDOUT "Done.\n\n";}


	}
	else 
	{	
		foreach my $speaker_dir (@exe_models) {
			
			my $hmmpath=$sv_system_path.$speaker_dir.$SR_lib::sign.$hmmBaseDir;
			&SR_lib::check_and_create_dir($hmmpath);
			
			my $hmmpath_zeroth=$hmmpath.$SR_lib::sign."hmm.0";
			&SR_lib::check_and_create_dir($hmmpath_zeroth);
			
			
			foreach my $hmm_model (@protos_exe) {
			  my $exec_string = "$prefixCommand ${SR_lib::htk_bin}HInit -i 5 $prompt_HTK_Command_String -m $extraOptions -M $hmmpath_zeroth -v $minVar -I ${sv_system_path}$mlf -S $sv_system_path$speaker_dir/$trainingModelSessionsList ${proto_path}${SR_lib::sign}$hmm_model";
			  $exit_state = system ($exec_string);
			  if ($exit_state) {
				print STDERR "Command $exec_string failed\n";
				exit 1;
			  }
			  #print ("$exec_string \n");
			
			}   
					 
			foreach my $statesType (keys %prototypesType) {
				my $list_length = scalar(@{$prototypesType{$statesType}});
				my $template_prototype = $prototypesType{$statesType}->[0];
				for (my $i=1; $i < $list_length; $i++) {
					my $new_prototype = $prototypesType{$statesType}->[$i];
					&SR_lib::create_prototype_from_template($hmmpath_zeroth, $new_prototype, $template_prototype);
				}
			}
	
		}


	}
	  


}
else #World Model -> gemeinsames Modell fr alle Trainingsdaten
  {
    
    &SR_lib::check_and_create_dir($world_path);
	$world_path = ${world_path}.$SR_lib::sign.$hmmBaseDir.$SR_lib::sign;
    &SR_lib::check_and_create_dir($world_path);
    my $world_path_zeroth=$world_path."hmm.0";
    &SR_lib::check_and_create_dir($world_path_zeroth);
	
	&SR_lib::create_multispeaker_list($sv_system_name, [$trainingWorldConf]);
	
	my $multispeaker_list = &SR_lib::get_filename_multispeaker_list($sv_system_name);

    foreach my $hmm_model (@protos_exe) {
      
		$exit_state = system ("$prefixCommand ${SR_lib::htk_bin}HInit -i 5 $prompt_HTK_Command_String -m $extraOptions -M $world_path_zeroth -v $minVar -I ${sv_system_path}$mlf -S ${multispeaker_list} ${proto_path}${SR_lib::sign}$hmm_model");
		if ($exit_state) {
			print STDERR "Command failed\n";
			exit 1;
		}
    
    
  	} 
  	
	&SR_lib::remove_multispeaker_list($sv_system_name);
  	
  	foreach my $statesType (keys %prototypesType) {
  		my $list_length = scalar(@{$prototypesType{$statesType}});
  		my $template_prototype = $prototypesType{$statesType}->[0];
		for (my $i=1; $i < $list_length; $i++) {
			my $new_prototype = $prototypesType{$statesType}->[$i];
			if ($SR_lib::verbose_level) { print STDOUT ("Creating flat model $new_prototype from $template_prototype\n");}
			&SR_lib::create_prototype_from_template($world_path_zeroth, $new_prototype, $template_prototype);
		}
	}


} 

if (! $distributedClient) {
	if ($onFinish) {
		if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onFinish\" ...\n");}
		$local_exit_code = system($onFinish);
	}
}


if (! $distributedClient) {
	&SR_lib::check_and_create_dir($log_path);
	my $log_handle = &SR_lib::open_file(">>", "$log_path" . "$log_file");
	print $log_handle "\n\n";
	print $log_handle "$start_time \n";
	print $log_handle "Running run_HInit.pl .....\n";
	print $log_handle "---------------------------\n";
	print $log_handle "Parameters:\n";
	print $log_handle "sv_system: $sv_system_path\n";
	print $log_handle "Options\n";
	print $log_handle "useGMM      : $useGMM\n";
	print $log_handle "varFloor    : $minVar\n";
	print $log_handle "p           : $parallelComputation\n";
	print $log_handle "w           : $buildWorldModel\n";
	if ($buildWorldModel) {
		print $log_handle "wl          : $trainingWorldConf\n";
	} else {
		print $log_handle "tc          : $trainingModelConf\n";
	}
	print $log_handle "protoDir          : $protoDir\n";
	print $log_handle "mlf               : $mlf\n";
	print $log_handle "hmmBaseDir        : $hmmBaseDir\n";
	print $log_handle "onStartup         : $onStartup\n";
	print $log_handle "onFinish          : $onFinish\n";
	print $log_handle "prefixCommand     : $prefixCommand\n";
	print $log_handle "extraOptions      : $extraOptions\n";
	print $log_handle "logfile           : $log_file\n";
	print $log_handle "cvs_version       : $CVS_Version_String\n";
	print $log_handle "cvs_name          : $CVS_Name_String\n";
	print $log_handle "Finishing... ",scalar localtime, "\n";

	if ($exit_state) {
		print $log_handle "\n Command failed !!!\n";
	}
	
	close $log_handle;
}

exit $exit_state;




__END__

=head1 NAME

run_HInit.pl  - run HInit for a SV system (useful for GMM models)

=head1 SYNOPSIS

run_HInit.pl [options] sv_system_name

Run HInit for SV system <sv_system_name>. Flat start hmms are written to the
directory "hmm.0" located in each speaker's directory. 

 Options:
 
 -w | world                do a flat start for the world model.
                           Default: not activated
 
 -wl | world_list          configuration for world flat start;
                           default is 'world_set/aa';
                           ignored, when -w option not set.
 
 -tc | trainingModelConf   configuration for normal flat start
                           default is 'training_set/at';
                           see "sub processConfigurations" in SR_lib.pm;
                           ignored, when -w option set.

 -v | minVar               minimum value for the elements of the covariance matrix
                           of the model. Default value is 0.001

 -m | mlf                  specify mlf file, default is "mlf_prompted.mlf"
                          
 -protoDir                 use models in directory <protoDir> as base.
 
 -hmmBaseDir               build models in given directory in 
                           speaker resp. world directory;
                           default is "hmm"

 -useGMM                   adapt behaviour for using a single state model 
                           with multiple mixtures (GMM).
 
 -xo | extraOptions        pass extra options to HInit

 -onStartup=s              command to perform before running on parallel host

 -onFinish=s               command to perform after running on parallel host

 -prefixCommand=s          command to perform before running HInit

 -p | parallelComputation  use several hosts for running HInit
                           (useful only for normal training, option -tc)
                          
 -logfile                  set logfile; default is log/log.txt.

 -? | help                 display this message.
 
=cut

