## ---- eval=TRUE, results='hide', message=F, warning=F--------------------
library(emuR)

# create demo data in directory provided by the tempdir() function
# (of course other directory paths may be chosen)
create_emuRdemoData(dir = tempdir())

# create path to demo data directory, which is
# called "emuR_demoData"
demoDataDir = file.path(tempdir(), "emuR_demoData")

# show demo data directories
list.dirs(demoDataDir, recursive = F, full.names = F)

# create path to TextGrid collection
tgColDir = file.path(demoDataDir, "TextGrid_collection")

# show content of TextGrid_collection directory
list.files(tgColDir)


# convert TextGrid collection to the emuDB format
convert_TextGridCollection(dir = tgColDir,
                           dbName = "myFirst",
                           targetDir = tempdir(),
                           tierNames = c("Text", "Syllable",
                                         "Phoneme", "Phonetic"))
# get path to emuDB called "myFirst"
# that was created by convert_TextGridCollection()
path2directory = file.path(tempdir(), "myFirst_emuDB")

# load emuDB into current R session
dbHandle = load_emuDB(path2directory, verbose = FALSE)


summary(dbHandle)

# list level definitions
# as this reveals the "-autobuildBackup" levels
# added by the autobuild_linkFromTimes() calls
list_levelDefinitions(dbHandle)


## ---- results='hide', message=F, warning=F-------------------------------
list_linkDefinitions(dbHandle)

## ----results='hide', message=F, warning=F--------------------------------
# invoke autobuild function
# for "Text" and "Syllable" levels
autobuild_linkFromTimes(dbHandle,
                        superlevelName = "Text",
                        sublevelName = "Syllable",
                        convertSuperlevel = TRUE,
                        newLinkDefType = "ONE_TO_MANY")

# invoke autobuild function
# for "Syllable" and "Phoneme" levels
autobuild_linkFromTimes(dbHandle,
                        superlevelName = "Syllable",
                        sublevelName = "Phoneme",
                        convertSuperlevel = TRUE,
                        newLinkDefType = "ONE_TO_MANY")

# invoke autobuild function
# for "Phoneme" and "Phonetic" levels
autobuild_linkFromTimes(dbHandle,
                        superlevelName = "Phoneme",
                        sublevelName = "Phonetic",
                        convertSuperlevel = TRUE,
                        newLinkDefType = "MANY_TO_MANY")

# list level definitions
list_levelDefinitions(dbHandle)

list_linkDefinitions(dbHandle)

# remove the levels containing the "-autobuildBackup"
# suffix
remove_levelDefinition(dbHandle,
                       name = "Text-autobuildBackup",
                       force = TRUE,
                       verbose = FALSE)

remove_levelDefinition(dbHandle,
                       name = "Syllable-autobuildBackup",
                       force = TRUE,
                       verbose = FALSE)

remove_levelDefinition(dbHandle,
                       name = "Phoneme-autobuildBackup",
                       force = TRUE,
                       verbose = FALSE)

# list level definitions
list_levelDefinitions(dbHandle)

list_linkDefinitions(dbHandle)

#serve(dbHandle)

## ----results='hide', message=F, warning=F--------------------------------
get_legalLabels(dbHandle,
                levelName = "Syllable",
                attributeDefinitionName = "Syllable")

set_legalLabels(dbHandle,
                levelName = "Syllable",
                attributeDefinitionName = "Syllable",
                legalLabels = c("S", "W"))
#serve(dbHandle)

## ----results='hide', message=F, warning=F--------------------------------
add_labelGroup(dbHandle, 
               name = "Vowels",
               values = c("i:", "o:", "V"))

list_labelGroups(dbHandle)

#####Delete this again with ...
#remove_labelGroup(dbHandle, 
#               name = "Vowels")

query (emuDBhandle = dbHandle,
       query = "Phonetic == Vowels")
#instead of
query (emuDBhandle = dbHandle,
       query = "Phonetic == i: | o: | V")


## ---- eval=FALSE---------------------------------------------------------
## list_attrDefLabelGroups(dbHandle,levelName = "Phonetic",attributeDefinitionName = "Phonetic")
## 
## # we could change this with
## #add_attributeDefinition()
## add_attributeDefinition(dbHandle,levelName = "Phonetic",name="IPA")
## 
## 
## #serve(ae)
## #however, IPA is empty (has no labels yet)
## remove_attributeDefinition(dbHandle,levelName = "Phonetic",name="IPA",force=TRUE)
## #Better:
## duplicate_level(dbHandle, levelName = "Phonetic",duplicateLevelName = "IPA")
## summary(ae)
## #manually replace labels (here only in two cases)
## replace_itemLabels(dbHandle,attributeDefinitionName = "IPA",origLabels = c("V","E"),newLabels = c("ʌ","ɛ"))
## 

## ----results='hide', message=F, warning=F--------------------------------
vignette()

## ---- eval=FALSE, results='hide', message=F, warning=F-------------------
## vignette("emuDB")

## ---- echo=FALSE, results='hide', message=FALSE--------------------------
# clean up emuR_demoData (in case it exists)
unlink(file.path(tempdir(), "emuR_demoData"), recursive = TRUE)
unlink(file.path(tempdir(), "myEmuDatabase_emuDB"), recursive = TRUE)

